<?php

namespace App\Http\Controllers\Admin;

use Auth;
use App\User;
use App\Language;
use App\Genres;
use App\ActorDirector;
use App\Http\Requests;
use Illuminate\Http\Request;
use Session;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Exception;

class ImportImdbShowController extends MainAdminController
{
    private $tmdbApiKey;
    private $defaultLanguage;
    private $baseUrl = 'https://api.themoviedb.org/3';
    
    public function __construct()
    {
        $this->middleware('auth');
        $this->tmdbApiKey = getcong('tmdb_api_key');
        $this->defaultLanguage = set_tmdb_language();
    }

    /**
     * Find and import TV show data from TMDB using TMDB ID or IMDb ID
     */
    public function find_imdb_show(Request $request)
    {
        try {
            $showId = $request->get('id');
            //$isImdbId = $request->get('is_imdb_id', false); // Flag to determine if ID is IMDb ID
            
            if (!$showId) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Show ID is required'
                ]);
            }

            // Validate API key
            if (!$this->tmdbApiKey) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'TMDB API key not configured'
                ]);
            }

            // Get TMDB ID (either directly or by converting from IMDb ID)
            $tmdbId = $this->getTmdbIdFromImdb($showId);
            
            if (!$tmdbId) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Show not found'
                ]);
            }

            // Fetch show data
            $showData = $this->fetchShowData($tmdbId);
            
            if (!$showData) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Failed to fetch show data'
                ]);
            }

            // Process and format the response
            $response = $this->processShowData($showData, $tmdbId);
            
            return response()->json($response);

        } catch (Exception $e) {
            Log::error('TMDB Show Import Error: ' . $e->getMessage());
            
            return response()->json([
                'imdb_status' => 'fail',
                'message' => 'An error occurred while fetching show data'
            ]);
        }
    }

    /**
     * Convert IMDb ID to TMDB ID for TV shows
     */
    private function getTmdbIdFromImdb($imdbId)
    {
        $url = "{$this->baseUrl}/find/{$imdbId}?external_source=imdb_id&language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}";
          
        $response = $this->makeApiRequest($url);
        
        if ($response && isset($response->tv_results) && !empty($response->tv_results)) {
            return $response->tv_results[0]->id;
        }
        
        return null;
    }

    /**
     * Fetch TV show data from TMDB API
     */
    private function fetchShowData($tmdbId)
    {
        $url = "{$this->baseUrl}/tv/{$tmdbId}?append_to_response=videos,aggregate_credits&language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}";
        return $this->makeApiRequest($url);
    }

    /**
     * Make API request to TMDB
     */
    private function makeApiRequest($url)
    {
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer " . $this->tmdbApiKey,
                "accept: application/json",
                "User-Agent: TVApp/1.0"
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_FOLLOWLOCATION => true
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $error = curl_error($curl);
        
        curl_close($curl);

        if ($error) {
            Log::error("cURL Error: " . $error);
            return null;
        }

        if ($httpCode !== 200) {
            Log::error("HTTP Error: " . $httpCode . " for URL: " . $url);
            return null;
        }

        $result = json_decode($response);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::error("JSON Decode Error: " . json_last_error_msg());
            return null;
        }

        return $result;
    }

    /**
     * Process TV show data and return formatted response
     */
    private function processShowData($showData, $tmdbId)
    {
        $response = [
            'imdb_status' => 'success',
            'imdbid' => $showData->external_ids->imdb_id ?? '',
            'tmdb_id' => $tmdbId,
            'imdb_rating' => isset($showData->vote_average) ? round($showData->vote_average, 1) : 0,
            'imdb_votes' => $showData->vote_count ?? 0,
            'title' => $showData->original_name ?? $showData->name ?? '',
            'plot' => $showData->overview ?? '',
            'first_air_date' => $showData->first_air_date ?? '',
            'last_air_date' => $showData->last_air_date ?? '',
            'number_of_seasons' => $showData->number_of_seasons ?? 0,
            'number_of_episodes' => $showData->number_of_episodes ?? 0,
            'status' => $showData->status ?? '',
            'type' => $showData->type ?? '',
        ];

        // Process language
        $response['language'] = $this->processShowLanguage($showData);

        // Process genres
        $response['genre'] = $this->processShowGenres($showData);

        // Process cast and crew
        $this->processShowCredits($showData, $response);

        // Process images
        $this->processShowImages($showData, $response);

        // Process trailer
        $response['trailer_url'] = $this->processShowTrailer($showData);

        // Process additional show-specific data
        $this->processShowNetworks($showData, $response);

        return $response;
    }

    /**
     * Process TV show language
     */
    private function processShowLanguage($showData)
    {
        if (!isset($showData->spoken_languages) || empty($showData->spoken_languages)) {
            return null;
        }

        $langName = $showData->spoken_languages[0]->english_name ?? '';
        return $langName ? Language::getLanguageID($langName) : null;
    }

    /**
     * Process TV show genres
     */
    private function processShowGenres($showData)
    {
        $genres = [];
        
        if (isset($showData->genres) && is_array($showData->genres)) {
            foreach ($showData->genres as $genre) {
                if (isset($genre->name)) {
                    $genreId = Genres::getGenresID($genre->name);
                    if ($genreId) {
                        $genres[] = $genreId;
                    }
                }
            }
        }
        
        return $genres;
    }

    /**
     * Process TV show credits (cast and crew)
     */
    private function processShowCredits($showData, &$response)
    {
        // Get aggregate credits if available, otherwise try regular credits
        $credits = $showData->aggregate_credits ?? null;
        
        if (!$credits) {
            // Fallback: fetch regular credits
            $creditsData = $this->makeApiRequest("{$this->baseUrl}/tv/{$showData->id}/credits?language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}");
            $credits = $creditsData;
        }

        if ($credits) {
            $response['actors'] = $this->processShowActors($credits);
            $response['director'] = $this->processShowDirectors($credits);
        } else {
            $response['actors'] = [];
            $response['director'] = [];
        }
    }

    /**
     * Process TV show actors
     */
    private function processShowActors($credits)
    {
        $actorOptions = [];
        
        if (!isset($credits->cast) || !is_array($credits->cast)) {
            return $actorOptions;
        }

        $castMembers = array_slice($credits->cast, 0, 20); // Limit to first 20 for TV shows
        
        foreach ($castMembers as $castMember) {
            try {
                $actorId = $this->processActorDirector($castMember, 'actor');
                if ($actorId) {
                    $actorName = $castMember->original_name ?? $castMember->name ?? '';
                    $actorOptions[] = "<option value='{$actorId}' selected>{$actorName}</option>";
                }
            } catch (Exception $e) {
                Log::warning("Error processing TV show actor: " . $e->getMessage());
                continue;
            }
        }
        
        return $actorOptions;
    }

    /**
     * Process TV show directors
     */
    private function processShowDirectors($credits)
    {
        $directorOptions = [];
        
        if (!isset($credits->crew) || !is_array($credits->crew)) {
            return $directorOptions;
        }

        foreach ($credits->crew as $crewMember) {
            try {
                // For TV shows, check both 'job' and 'jobs' array
                $isDirector = false;
                
                if (isset($crewMember->job) && $crewMember->job === 'Director') {
                    $isDirector = true;
                } elseif (isset($crewMember->jobs) && is_array($crewMember->jobs)) {
                    foreach ($crewMember->jobs as $job) {
                        if (isset($job->job) && $job->job === 'Director') {
                            $isDirector = true;
                            break;
                        }
                    }
                }

                if ($isDirector) {
                    $directorId = $this->processActorDirector($crewMember, 'director');
                    if ($directorId) {
                        $directorName = $crewMember->original_name ?? $crewMember->name ?? '';
                        $directorOptions[] = "<option value='{$directorId}' selected>{$directorName}</option>";
                    }
                }
            } catch (Exception $e) {
                Log::warning("Error processing TV show director: " . $e->getMessage());
                continue;
            }
        }
        
        return $directorOptions;
    }

    /**
     * Process actor or director and save to database if not exists
     */
    private function processActorDirector($person, $type)
    {
        $name = $person->original_name ?? $person->name ?? '';
        $tmdbId = $person->id ?? null;
        
        if (!$name || !$tmdbId) {
            return null;
        }

        // Check if actor/director already exists
        $existingPerson = ActorDirector::where('ad_name', $name)
                                     ->where('ad_type', $type)
                                     ->first();

        if ($existingPerson) {
            return $existingPerson->id;
        }

        // Fetch detailed person information
        $personDetails = $this->fetchPersonDetails($tmdbId);
        
        if (!$personDetails) {
            return null;
        }

        // Create new actor/director record
        $actorDirector = new ActorDirector();
        $actorDirector->ad_type = $type;
        $actorDirector->ad_name = addslashes($name);
        $actorDirector->ad_bio = addslashes($personDetails->biography ?? '');
        $actorDirector->ad_birthdate = $personDetails->birthday ? strtotime($personDetails->birthday) : null;
        $actorDirector->ad_place_of_birth = addslashes($personDetails->place_of_birth ?? '');
        $actorDirector->ad_tmdb_id = $tmdbId;
        $actorDirector->ad_slug = Str::slug($name, '-', null);

        // Download and save profile image
        if (isset($person->profile_path) && $person->profile_path) {
            $imagePath = $this->downloadPersonImage($person->profile_path);
            if ($imagePath) {
                $actorDirector->ad_image = $imagePath;
            }
        }

        $actorDirector->save();
        
        return $actorDirector->id;
    }

    /**
     * Fetch person details from TMDB
     */
    private function fetchPersonDetails($personId)
    {
        $url = "{$this->baseUrl}/person/{$personId}?language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}";
        return $this->makeApiRequest($url);
    }

    /**
     * Download person profile image
     */
    private function downloadPersonImage($profilePath)
    {
        try {
            $imageUrl = 'https://image.tmdb.org/t/p/w300' . $profilePath;
            $fileName = basename(parse_url($imageUrl, PHP_URL_PATH));
            $savePath = public_path('/upload/images/' . $fileName);
            
            if (grab_image($imageUrl, $savePath)) {
                return 'upload/images/' . $fileName;
            }
        } catch (Exception $e) {
            Log::warning("Error downloading person image: " . $e->getMessage());
        }
        
        return null;
    }

    /**
     * Process TV show images
     */
    private function processShowImages($showData, &$response)
    {
        // Process poster
        if (isset($showData->poster_path) && $showData->poster_path) {
            $posterPath = 'https://image.tmdb.org/t/p/w300' . $showData->poster_path;
            $response['thumbnail'] = $posterPath;
            $response['thumbnail_name'] = basename(parse_url($posterPath, PHP_URL_PATH));
        }

        // Process backdrop
        if (isset($showData->backdrop_path) && $showData->backdrop_path) {
            $backdropPath = 'https://image.tmdb.org/t/p/w780' . $showData->backdrop_path;
            $response['poster'] = $backdropPath;
            $response['poster_name'] = basename(parse_url($backdropPath, PHP_URL_PATH));
        }
    }

    /**
     * Process TV show trailer
     */
    private function processShowTrailer($showData)
    {
        if (!isset($showData->videos->results) || !is_array($showData->videos->results)) {
            return '';
        }

        foreach ($showData->videos->results as $video) {
            if (isset($video->type) && $video->type === 'Trailer' && 
                isset($video->site) && $video->site === 'YouTube' && 
                isset($video->key)) {
                return 'https://www.youtube.com/watch?v=' . $video->key;
            }
        }

        return '';
    }

    /**
     * Process TV show networks and production companies
     */
    private function processShowNetworks($showData, &$response)
    {
        // Networks
        if (isset($showData->networks) && is_array($showData->networks)) {
            $response['networks'] = array_map(function($network) {
                return $network->name;
            }, $showData->networks);
        }

        // Production companies
        if (isset($showData->production_companies) && is_array($showData->production_companies)) {
            $response['production_companies'] = array_map(function($company) {
                return $company->name;
            }, $showData->production_companies);
        }

        // Created by
        if (isset($showData->created_by) && is_array($showData->created_by)) {
            $response['creators'] = array_map(function($creator) {
                return $creator->name;
            }, $showData->created_by);
        }
    }

    /**
     * Find and import TV episode data from TMDB
     */
    public function find_imdb_episode(Request $request)
    {
        try {
            $episodeId = $request->get('id');
            
            if (!$episodeId) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Episode ID is required'
                ]);
            }

            // Validate API key
            if (!$this->tmdbApiKey) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'TMDB API key not configured'
                ]);
            }

            // Find episode by IMDb ID
            $episodeData = $this->findEpisodeByImdbId($episodeId);
            
            if (!$episodeData) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Episode not found'
                ]);
            }

            // Process and format the response
            $response = $this->processEpisodeData($episodeData, $episodeId);
            
            return response()->json($response);

        } catch (Exception $e) {
            Log::error('TMDB Episode Import Error: ' . $e->getMessage());
            
            return response()->json([
                'imdb_status' => 'fail',
                'message' => 'An error occurred while fetching episode data'
            ]);
        }
    }

    /**
     * Find episode by IMDb ID
     */
    private function findEpisodeByImdbId($imdbId)
    {
        $url = "{$this->baseUrl}/find/{$imdbId}?external_source=imdb_id&language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}";
        $response = $this->makeApiRequest($url);
        
        if ($response && isset($response->tv_episode_results) && !empty($response->tv_episode_results)) {
            return $response->tv_episode_results[0];
        }
        
        return null;
    }

    /**
     * Process episode data and return formatted response
     */
    private function processEpisodeData($episodeData, $imdbId)
    {
        $response = [
            'imdb_status' => 'success',
            'imdbid' => $imdbId,
            'tmdb_id' => $episodeData->id ?? '',
            'show_id' => $episodeData->show_id ?? '',
            'imdb_rating' => isset($episodeData->vote_average) ? round($episodeData->vote_average, 1) : 0,
            'imdb_votes' => $episodeData->vote_count ?? 0,
            'title' => $episodeData->name ?? '',
            'plot' => $episodeData->overview ?? '',
            'runtime' => $episodeData->runtime ?? 0,
            'released' => isset($episodeData->air_date) ? date('m/d/Y', strtotime($episodeData->air_date)) : '',
            'season_number' => $episodeData->season_number ?? 0,
            'episode_number' => $episodeData->episode_number ?? 0,
            'production_code' => $episodeData->production_code ?? '',
        ];

        // Process episode image
        if (isset($episodeData->still_path) && $episodeData->still_path) {
            $stillPath = 'https://image.tmdb.org/t/p/w780' . $episodeData->still_path;
            $response['poster'] = $stillPath;
            $response['poster_name'] = basename(parse_url($stillPath, PHP_URL_PATH));
        }

        return $response;
    }

    /**
     * Get TV show seasons
     */
    public function get_show_seasons(Request $request)
    {
        try {
            $showId = $request->get('show_id');
            
            if (!$showId) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'Show ID is required'
                ], 400);
            }

            $showData = $this->makeApiRequest("{$this->baseUrl}/tv/{$showId}?language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}");
            
            if (!$showData) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'Show not found'
                ], 404);
            }

            $seasons = [];
            if (isset($showData->seasons) && is_array($showData->seasons)) {
                foreach ($showData->seasons as $season) {
                    $seasons[] = [
                        'id' => $season->id,
                        'season_number' => $season->season_number,
                        'name' => $season->name,
                        'episode_count' => $season->episode_count,
                        'air_date' => $season->air_date,
                        'overview' => $season->overview ?? '',
                        'poster_path' => $season->poster_path ? 'https://image.tmdb.org/t/p/w300' . $season->poster_path : null
                    ];
                }
            }

            return response()->json([
                'status' => 'success',
                'seasons' => $seasons
            ]);

        } catch (Exception $e) {
            Log::error('Get Seasons Error: ' . $e->getMessage());
            
            return response()->json([
                'status' => 'fail',
                'message' => 'An error occurred while fetching seasons'
            ]);
        }
    }

    /**
     * Get season episodes
     */
    public function get_season_episodes(Request $request)
    {
        try {
            $showId = $request->get('show_id');
            $seasonNumber = $request->get('season_number');
            
            if (!$showId || !$seasonNumber) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'Show ID and season number are required'
                ]);
            }

            $seasonData = $this->makeApiRequest("{$this->baseUrl}/tv/{$showId}/season/{$seasonNumber}?language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}");
            
            if (!$seasonData) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'Season not found'
                ]);
            }

            $episodes = [];
            if (isset($seasonData->episodes) && is_array($seasonData->episodes)) {
                foreach ($seasonData->episodes as $episode) {
                    $episodes[] = [
                        'id' => $episode->id,
                        'episode_number' => $episode->episode_number,
                        'name' => $episode->name,
                        'overview' => $episode->overview ?? '',
                        'air_date' => $episode->air_date,
                        'runtime' => $episode->runtime ?? 0,
                        'vote_average' => $episode->vote_average ?? 0,
                        'still_path' => $episode->still_path ? 'https://image.tmdb.org/t/p/w300' . $episode->still_path : null
                    ];
                }
            }

            return response()->json([
                'status' => 'success',
                'episodes' => $episodes,
                'season_info' => [
                    'name' => $seasonData->name,
                    'overview' => $seasonData->overview ?? '',
                    'air_date' => $seasonData->air_date,
                    'episode_count' => count($episodes)
                ]
            ]);

        } catch (Exception $e) {
            Log::error('Get Episodes Error: ' . $e->getMessage());
            
            return response()->json([
                'status' => 'fail',
                'message' => 'An error occurred while fetching episodes'
            ]);
        }
    }
}